// info_window.js

const valueMaps = {
    // Χώροι Στάθμευσης
    status: { 1: "Δημόσιο", 2: "Ιδιωτικό" },
    vehicle_type: { 1: "Συμβατικό", 2: "Ηλεκτρικό" },
    side_clearance_lr: { 1: "Αριστερά", 2: "Δεξιά", 3: "Και τα δύο" },

    // Διαβάσεις
    typos_odou: {
        1: "Τοπικός δρόμος / χαμηλής κυκλοφορίας",
        2: "Δρόμος συλλέκτης / μέτριας κυκλοφορίας",
        3: "Κύρια αρτηρία πόλης / υψηλής κυκλοφορίας",
        4: "Πεζόδρομος"
    },
    street_lights_type: {
        1: "cat's eye",
        2: "Τύπου LED",
        3: "Φωτεινές γραμμές"
    },

    // Ράμπες εισόδου κτιρίων
    door_type: {
        1: "Χειροκίνητη",
        2: "Συρόμενη",
        3: "Αυτόματη"
    },
    handrails: {
        0: "Όχι",
        1: "Μεταλλική",
        2: "Χτιστή",
        3: "Ξύλινη"
    }
};

function escapeHTML(str) {
    if (str === null || str === undefined) return '';
    return String(str)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#039;');
}

// Έλεγχος URL
function sanitizeUrl(url) {
    if (typeof url !== 'string') return '#';
    url = url.trim();
    if (/^(https?:\/\/|\/)/i.test(url)) return escapeHTML(url);
    return '#';
}

// Generate HTML

function generateInfoHTML(data) {
    const fieldLabels = fieldLabelsMap[data.category] || {};

    function getLabel(key, val) {
        if (val === null || val === undefined || val === '') return '';
        
        // Δημοτική Ενότητα από βάση
        if (key === 'dimotiki_enotita_id') {
            return dimotikiEnotitaMap[val] || '—';
        }

        if (valueMaps[key] && valueMaps[key][Number(val)] !== undefined)
            return valueMaps[key][Number(val)];

        if (fieldLabels[`${key}_${val}`] !== undefined)
            return fieldLabels[`${key}_${val}`];

        if (val === true || val === "1" || val === 1) {
            if (fieldLabels[`${key}_yes`] !== undefined)
                return fieldLabels[`${key}_yes`];
        }

        if (val === false || val === "0" || val === 0) {
            if (fieldLabels[`${key}_no`] !== undefined)
                return fieldLabels[`${key}_no`];
        }

        return escapeHTML(val);
    }

    let infoHTML = `
    
    <div style="
        max-width: 400px;    /* λιγότερο από πριν */
        min-width: 0;
        font-family: Arial, sans-serif;
        font-size:12px;
        line-height:1.4;
        color:#333;
    ">


        <h4 style="
            margin:0 0 8px 0;
            color:#003366;
            font-weight:bold;
            text-align:center;
        ">
        ${escapeHTML(data.category)}</h4>
        <table class="table table-hover table-borderless" style="
            width:100%;
            border-collapse:collapse;
            border-spacing:0;
            table-layout:auto;
            font-size:12px;
        ">
    `;

    const hiddenKeys = ['id','marker','category','category_id','dimos_id', 'lng', 'lat'];

    for (let key in data) {
        if (hiddenKeys.includes(key)) continue;

        let val = data[key];
        if (val instanceof File) val = val.name;

        if (val) {
            if (key === 'street_view') {
                val = `<a href="${sanitizeUrl(val)}" target="_blank" style="color:#0066cc; text-decoration:underline;">Προβολή</a>`;
            }
            else if (key === 'image1' || key === 'image2') {
                const imgUrl = sanitizeUrl(`/images/${val}`);
                val = `<a href="${imgUrl}" target="_blank" style="color:#0066cc; text-decoration:underline;">Προβολή</a>`;
            }

            else {
                val = getLabel(key, val);
            }
        } else {
            val = getLabel(key, val);
        }

        infoHTML += `
        <tr>
            <td style="color:#003366; font-weight:bold; padding:4px 2px; white-space:nowrap !important;">
                ${escapeHTML(fieldLabels[key] || key)}
            </td>
            <td style="padding:4px 2px !important;">
                ${val}  <!-- εδώ το val ήδη περνάει από escapeHTML ή sanitizeUrl -->
            </td>
        </tr>`;
    }

    //Edit και Delete κουμπιά 
    infoHTML += `
        </table>
        <div style="margin-top:6px; display:flex; gap:6px; justify-content:center;">
            <button class="btn btn-warning btn-sm editMarkerBtn" style="font-size:12px;">
                Επεξεργασία
            </button>
            <button class="btn btn-danger btn-sm deleteMarkerBtn" style="font-size:12px;">
                Διαγραφή
            </button>
        </div>
    `;

    return infoHTML;
}

// InfoWindow 
let currentInfoWindow = null;

function attachInfoWindow(marker, data, map) {

    marker.addListener('click', () => {

        // Κλείσιμο προηγούμενου InfoWindow, αν υπάρχει
        if (currentInfoWindow) currentInfoWindow.close();

        const infoWindow = new google.maps.InfoWindow({
            content: generateInfoHTML(data),
            maxWidth: 400
        });

        infoWindow.open(map, marker);
        currentInfoWindow = infoWindow;

        google.maps.event.addListener(infoWindow, 'domready', () => {

            //Edit Button
            const editBtn = document.querySelector('.editMarkerBtn');
            if (editBtn) {
                editBtn.onclick = () => openFormWithData(data);
            }

            // Delete Button
            const iw = document.querySelector('.gm-style-iw');
            if (!iw) return;

            const deleteBtn = iw.querySelector('.deleteMarkerBtn');
            if (!deleteBtn) return;

            deleteBtn.onclick = () => {

                const alertBox = document.getElementById('deleteAlertContainer');
                if (!alertBox) return;

                // Δημιουργία alert με Ναι και Όχι
                alertBox.innerHTML = `
                    <div class="alert alert-warning delete-alert" style="text-align:center; padding:15px;">
                        <div style="margin-bottom:10px;">Προσοχή! Είστε σίγουρος/η ότι θέλετε να διαγράψετε το marker;</div>
                        <div>
                            <button class="btn btn-danger btn-sm" id="confirmDeleteBtn" style="margin-right:8px;" >Διαγραφή</button>
                            <button class="btn btn-default btn-sm" id="cancelDeleteBtn" style="background-color: #ccd1d6; color: #333; font-weight: 700;">Ακύρωση</button>
                        </div>
                    </div>
                `;             

                // Κλείσιμο alert με Όχι
                document.getElementById('cancelDeleteBtn').onclick = () => {
                    alertBox.innerHTML = '';
                };

                // Επιβεβαίωση διαγραφής με Ναι
                document.getElementById('confirmDeleteBtn').onclick = async () => {
                    const category_id = categoryToIdMap[data.category];
                    if (!category_id) return;

                    try {
                        const res = await fetch('/va/delete_markers.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({ id: data.id, category_id })
                        });

                        const result = await res.json();
                        if (!result.success) throw new Error();

                        marker.setMap(null);
                        if (currentInfoWindow) currentInfoWindow.close();

                        alertBox.innerHTML = `
                            <div class="alert alert-success fade in">
                                ✅ Η εγγραφή διαγράφηκε επιτυχώς
                            </div>
                        `;

                        setTimeout(() => alertBox.innerHTML = '', 3000);

                    } catch (e) {
                        alertBox.innerHTML = `
                            <div class="alert alert-danger fade in">
                                ❌ Σφάλμα διαγραφής
                            </div>
                        `;
                    }
                };
            };
        });
    });
}
