// helpers.js

// Προ-συμπλήρωση πεδίων (inputs, selects, textarea, checkboxes)
function populateFields(fieldsDiv, data) {
    Object.keys(data).forEach(key => {
        const el = fieldsDiv.querySelector('#' + key);
        if (!el) return;

        let value = data[key];

        // Αν υπάρχει valueMaps για το πεδίο
        if (valueMaps[key] && valueMaps[key][value] !== undefined) {
            value = value; // το select παίρνει την ίδια τιμή (number/string)
        }

        // Συμπλήρωση ανά τύπο πεδίου
        if (el.tagName === 'SELECT') {
            el.value = value;
        } else if (el.type === 'checkbox') {
            el.checked = !!value;
        } else if (el.type !== 'file') {
            el.value = value;
        }
    });
}




// Δημιουργία/ενημέρωση πεδίων Lat/Lng
function setLatLngFields(fieldsDiv, category, lat, lng) {
    const idMap = {
        "Χώροι Στάθμευσης": ["parking_latitude", "parking_longitude"],
        "Διαβάσεις": ["crossing_latitude", "crossing_longitude"],
        "Ράμπες Πεζοδρομίων": ["ped_ramp_latitude", "ped_ramp_longitude"],
        "Ράμπες Εισόδου Κτιρίων": ["building_ramp_latitude", "building_ramp_longitude"]
    };

    const [latId, lngId] = idMap[category] || [];
    if (!latId) return;

    [[latId, lat], [lngId, lng]].forEach(([id, value]) => {
        let input = fieldsDiv.querySelector('#' + id);
        if (!input) {
            const label = document.createElement('label');
            label.textContent = id.includes('lat') ? 'Latitude' : 'Longitude';
            input = document.createElement('input');
            input.type = 'text';
            input.id = id;
            input.className = 'form-control';
            input.readOnly = true;
            fieldsDiv.appendChild(label);
            fieldsDiv.appendChild(input);
        }
        input.value = value ? parseFloat(value).toFixed(6) : '';
    });
}

/**
 * Συνδέει ένα InfoWindow με κουμπί Edit σε ένα marker
 * @param {google.maps.Marker} marker - Ο marker που θα έχει InfoWindow
 * @param {Object} data - Τα δεδομένα του marker
 */
function attachInfoWindow(marker, data) {
    if (!marker || !data) return;

    // Δημιουργεί το InfoWindow με το HTML από generateInfoHTML
    const infoWindow = new google.maps.InfoWindow({
        content: generateInfoHTML(data)
    });

    // Όταν κάνουμε click στο marker
    marker.addListener('click', () => {
        infoWindow.open(marker.getMap(), marker);

        // Όταν το InfoWindow είναι έτοιμο στο DOM
        google.maps.event.addListenerOnce(infoWindow, 'domready', () => {
            const editBtn = document.querySelector('.gm-style-iw .editMarkerBtn');
            if (!editBtn) return;

            // Κλωνοποίηση για να αφαιρέσουμε προηγούμενους listeners
            const newBtn = editBtn.cloneNode(true);
            editBtn.replaceWith(newBtn);

            newBtn.addEventListener('click', () => {
                infoWindow.close();          // Κλείνουμε το InfoWindow
                openFormWithData(data);      // Ανοίγουμε τη φόρμα με τα δεδομένα
            });
        });
    });
}


// Λαμβάνει δεδομένα από μια φόρμα
function getFormData(fieldsDiv) {
    const data = {};
    [...fieldsDiv.querySelectorAll('input, textarea, select')].forEach(el => {
        if (el.type === 'checkbox') data[el.id] = el.checked;
        else if (el.type === 'file') data[el.id] = el.files[0]?.name || null;
        else data[el.id] = el.value || '';
    });
    return data;
}




// Άνοιγμα φόρμας με δεδομένα από server
function openFormFromServerData(id, category) {
    fetch(`/va/get_marker.php?id=${id}&category=${category}`)
        .then(res => res.json())
        .then(data => {
            if (!data) return;

            const container = document.querySelector('#markerFormContainer');

            // 1. Δημιουργούμε τα πεδία της φόρμας ανά κατηγορία
            container.innerHTML = getTabFieldsHTML(category);

            // 2. Προ-συμπλήρωση πεδίων
            populateFields(container, data);

            // 3. Lat/Lng
            setLatLngFields(container, category, data.lat || 0, data.lng || 0);

            // 4. Εμφάνιση φόρμας
            container.style.display = 'block';
        })
        .catch(err => console.error(err));
}

// Παίρνει icon για marker
function getMarkerIcon(category) {
    const icons = {
        "Χώροι Στάθμευσης": "icons_map/parking-sign_pin.png",
        "Διαβάσεις": "icons_map/ped_pin2.png",
        "Ράμπες Πεζοδρομίων": "icons_map/ramp_pin3.png",
        "Ράμπες Εισόδου Κτιρίων": "icons_map/ramp_16_pin.png"
    };
    return icons[category] || "icons_map/default.png";
}


// draggable form
// draggable form (SAFE + BOUNDED)
function makeDraggable(element, handleOrSelector = null) {
    const handle = typeof handleOrSelector === 'string'
        ? element.querySelector(handleOrSelector)
        : handleOrSelector || element;

    if (!handle) {
        console.warn('makeDraggable: handle not found', handleOrSelector);
        return;
    }

    let isDragging = false;
    let offsetX = 0, offsetY = 0;

    handle.addEventListener('mousedown', (e) => {
        isDragging = true;

        const rect = element.getBoundingClientRect();
        offsetX = e.clientX - rect.left;
        offsetY = e.clientY - rect.top;

        document.body.style.userSelect = 'none';
        e.preventDefault();
    });

    document.addEventListener('mousemove', (e) => {
        if (!isDragging) return;

        let left = e.clientX - offsetX;
        let top  = e.clientY - offsetY;

        const winW = window.innerWidth;
        const winH = window.innerHeight;
        const rect = element.getBoundingClientRect();

        // boundaries
        if (left < 0) left = 0;
        if (top < 0) top = 0;
        if (left + rect.width > winW) left = winW - rect.width;
        if (top + rect.height > winH) top = winH - rect.height;

        element.style.left = left + 'px';
        element.style.top  = top + 'px';
    });

    document.addEventListener('mouseup', () => {
        isDragging = false;
        document.body.style.userSelect = '';
    });
}



//validateFields
// έλεγχος αν τα πεδία είναι σωστά συμπληρωμένα

function validateFields(fieldsDiv) {
    let valid = true;

    // Καθαρίζουμε προηγούμενα μηνύματα σφάλματος
    fieldsDiv.querySelectorAll('.error-message').forEach(el => el.remove());

    // Παίρνουμε όλα τα inputs και selects
    const inputs = fieldsDiv.querySelectorAll('input, select, textarea');

    inputs.forEach(input => {
        if(input.required && !input.value.trim()) {
            valid = false;
            const msg = document.createElement('span');
            msg.className = 'error-message';
            msg.textContent = 'Το πεδίο πρέπει να συμπληρωθεί σωστά';
            input.parentNode.appendChild(msg);
        }

        // Έλεγχος pattern (π.χ. πινακίδα)
        if(input.pattern) {
            const regex = new RegExp(input.pattern);
            if(input.value && !regex.test(input.value)) {
                valid = false;
                const msg = document.createElement('span');
                msg.className = 'error-message';
                msg.textContent = 'Μη έγκυρη μορφή';
                input.parentNode.appendChild(msg);
            }
        }
    });

    return valid;
}
